#include "validation.h"

#include <algorithm>
#include <array>
#include <filesystem>
#include <fstream>
#include <string>

// State file, if present, will contain:
// Original input on the first line
// Team output from the first pass on the second line
// Number of the current pass on the last line.
struct State {
    int number;
    std::array<int, 30> first_pass_output;
    int pass_number;
};

State read_state(const std::filesystem::path& state_file) {
    State state;
    std::ifstream state_stream(state_file);
    state_stream >> state.number;
    for(int i = 0; i < 30; ++i) {
        state_stream >> state.first_pass_output[i];
    }
    state_stream >> state.pass_number;
    return state;
}

void write_state(const std::filesystem::path& state_file, const State& state) {
    std::ofstream state_stream(state_file);
    state_stream << state.number << '\n';
    for(int i = 0; i < 30; ++i) {
        state_stream << state.first_pass_output[i] << ' ';
    }
    state_stream << '\n' << state.pass_number << std::endl;
}

constexpr int kPassCount = 10;

std::pair<int, int> select_numbers_for_pass(const std::array<int, 30> numbers, int pass) {
    if(pass == 1) {
        // Pick smallest
        return {numbers[0], numbers[1]};
    } else if(pass == 2) {
        // Pick larges
        return {numbers[numbers.size() - 2], numbers[numbers.size() - 1]};
    } else if(pass == 3) {
        // Pick smallest and largest
        return {numbers[0], numbers[numbers.size() - 1]};
    } else {
        // Pick random (also at pass 0 to have random samples)
        std::mt19937_64 gen(pass);
        int first  = Random::select(numbers.begin(), numbers.end(), gen);
        int second = Random::select(numbers.begin(), numbers.end(), gen);
        while(first == second) {
            second = Random::select(numbers.begin(), numbers.end(), gen);
        }
        return {first, second};
    }
}

int main(int argc, char* argv[]) {
    std::ifstream in(argv[1]);
    OutputValidator v(argc, argv);
    v.set_WA_handler([&]() {
        std::ofstream judgemessage{std::filesystem::path(argv[3]) / "judgemessage.txt"};
        judgemessage << "WA" << std::endl;
    });

    std::filesystem::path state_file    = std::filesystem::path(argv[3]) / "state.txt";
    std::filesystem::path nextpath_file = std::filesystem::path(argv[3]) / "nextpass.in";

    std::string pass;
    in >> pass;
    if(pass == "send") {
        // first pass

        int a;
        in >> a;
        State state;
        state.number = a;
        for(int i = 0; i < 30; ++i) {
            state.first_pass_output[i] = v.read_integer("first_pass_output", 1, 1000);
        }
        std::sort(state.first_pass_output.begin(), state.first_pass_output.end());
        state.pass_number = 0;

        for(int i = 0; i + 1 < 30; ++i) {
            v.check(state.first_pass_output[i] != state.first_pass_output[i + 1],
                    "Duplicate number in output ", state.first_pass_output[i]);
        }

        write_state(state_file, state);

        auto next_numbers = select_numbers_for_pass(state.first_pass_output, state.pass_number);
        std::ofstream nextpass{nextpath_file};
        nextpass << "receive\n" << next_numbers.first << " " << next_numbers.second << std::endl;
    } else {
        int output_number = v.read_integer("number", 1, 600);

        State state = read_state(state_file);
        state.pass_number++;
        v.check(output_number == state.number,
                "Output number does not match the original number. Expected ", state.number,
                " but got ", output_number);

        if(state.pass_number <= kPassCount) {
            auto next_numbers = select_numbers_for_pass(state.first_pass_output, state.pass_number);
            std::ofstream nextpass{nextpath_file};
            nextpass << "receive\n"
                     << next_numbers.first << " " << next_numbers.second << std::endl;

            write_state(state_file, state);
        } else {
            std::remove(nextpath_file.c_str());
        }
    }
    return 0;
}
